// en0c.c  taper and gee versus height
// gcc -o en0c en0c.c -lm ; ./en0c > en0c.dat ; gp en0c.gp
//
// KHL 2017 Aug 9
//    /usr/local/bin/gp adds bitstream-vera to GDFONTPATH then calls /usr/bin/gnuplot
//    
// 
// adjusted for 22.2 tonnes on the climber plus the starting force;
// 44 kN of acoustic force (assumed constant),  60 kN of bias force

// integrate the cross section versus height
//
// Start with 1 N at GEO, a very slender space elevator.
//
// the taper function is T(r) = exp( -(energy-energyGEO)/YURI ) )
// where energy = 0.5*om2*r*r+MU/r
//
// The cross section C(r) = C0 * taper(r) where C0 = 1/strength,
// and strength = MY * density = 27.077 MY * 1300kg/m3 = 35.200 GPa ?
// or  strength = MY * density = 27.000 MY * 1300kg/m3 = 35.100 GPa ?
// use the former, matches Assessment page 32

// cross section C(r) = taper(r)
// volume C(r) dR oer meter
// The cross section at apex is C(apex) = taper(apex)*C(geo)
//
// Page 32 and 33 tables:
// 38 MYuri      Rated Tether  [ 49.4 GPa @ 1300 kg/m³]
// 1300 kg/m³    Density of material
// 1.4		 Safety factor
// 35.2 GPa      Operational capacity 
// 27 MYuri      Operational capacity
// 6		 Taper Ratio
// 62.8 mm² 	 GEO cross section
// 10.5 mm²      Earth cross section
// 6000 kg	 Climber structure (including solar cells)
// 14000 kg	 Climber payload
// 6.3e6 kg	 Tether Mass
// 1.9e6 kg	 Apex Anchor
// 100,000 km	 Tether Apex radius
//
// 7 days climbers
// 1.00           6378 km
// 0.25          12756 km
// 0.10          19134 km
// 0.043	 26600 km MEO
// 0.041	 27000 km
// 0.014         34000 km
// 0.000         42160 km GEO
//
//
// Page 75:  11.8 MW Array 
//	 ??	V1	V2	V3
//	 50K	11.8M	11.8M	11.8M	Array Power
//	 10%	10%	40%	70%	PV efficiency 
//	 32	7400	3610	2400	kg
//	 400	590k	290k	191k	m² 
//
//	 0.64	0.63	0.31	0.20	kg/kW
//
// Baloney.  The SSPS community assumes 7kg/kW transmitted,
// server sky assumes 1 kg/kw locally IN ZERO G  
// 
// The 400 m²  number seems to be for the DLR solar SAIL,
// described here:  https://tinyurl.com/dlrsail
// http://www.dlr.de/fa/Portaldata/17/Resources/dokumente/institut/2002/2002_04.pdf
//
// That is a MIRROR, not an electrical device, and produces
// ZERO watts of power, and 3.6 milliNewtons of light pressure
// thrust in ZERO gravity, an acceleration of 10 m/s per DAY.  


#define ALT0         0.0     // km starting
#define ALT1     35767.0     // km end
#define ASTEP       20.0     // km step
#define ISTEP      0.125     // km integration step 
                       
#define APEX    300000.0     // km apex RADIUS  (gravity siphon?)

#define RE        6378.0     // km
#define MU      398600.4418  // km3/s2
#define SDAY     86164.0905  // sec siderial day
#define KM        1000.0     // meters per kilometer
#define YURI        27.0     // MegaYuris
#define F0       44000.0     // N starting force

#define TENSILE    1.0e12    // Pa
#define SOUND    28000.00    // m/s (alternate 1.0192e12 Pa)

#define DENS      1300.00    // kg/m3
#define CRGEO       62.8e-6  // m2 cross section at geo
#define CRGND       10.5e-6  // m2 cross section at ground
#define TRAT     6.01034     // taper ratio for output scaling

#define HOUR         3.6     // kiloseconds
#define MV       16000.0     // vehicle mass kg

#define SPMAX      100.0     // m/s
#define RT          24.0     // hours between vehicles


#include <stdio.h>
#include <math.h>

int main() {
   double  pi    = 4.0*atan(1.0)       ;
   double  deg   = 180.0/pi            ; // degrees per radian
   double  omega = 2.0*pi/SDAY         ; // rad/s angular velocity
   double  om2   = omega*omega         ; // angular velocity squared
   double  mu    = MU                  ; // angular velocity squared
   double  th    = 1.0 / 3.0           ;
   double  rg    = pow((mu/om2),th)    ; // km    geo altitude
   double  ra    = APEX                ; // km   apex altitude
   double  rmax  = rg - 0.001          ; // km  maximum altitude

   // not orbital energy, we have help from tether
   double  ee    = 0.5*om2*RE*RE+MU/RE ; // -MJ/kg energy at re
   double  eg    = 0.5*om2*rg*rg+MU/rg ; // -MJ/kg energy at geo
   double  ea    = 0.5*om2*ra*ra+MU/ra ; // -MJ/kg energy at apex

   double  ege   = ee-eg               ; // MJ/kg normalizing energy
   double  ega   = ea-eg               ; // MJ/kg anchor energy

   // taper
   double  tge   = exp( -ege / YURI )  ; // taper ratio to earth
   double  tga   = exp( -ega / YURI )  ; // taper ratio to apex

   double  ae    = MU/(RE*RE)-om2*RE   ; // km/s²  gravity at RE (down)
   double  aa    = om2*ra - MU/(ra*ra) ; // km/s²  anchor acceleration (up)

   double  yuri  = 1e6 * YURI          ; // m² /s²  yuris not Megayuris
   double  str   = yuri * DENS         ; // N/m²  strength
   double  tgeo  = str * CRGEO         ; // N tension at GEO
   double  tgeok = 1e-3 * tgeo         ; // kN tension at GEO
   double  tapxk = tga * tgeok         ; // kN strength
   double  crapx = tga * CRGEO         ; // m²  cross section at apex
   double  mapx  = tapxk / (1e3*aa)    ; // tonne apex 

   double  sound = sqrt(TENSILE/DENS)  ; // m/s speed of sound in tether
//         sound = SOUND               ; // m/s alternate for easy units

// GROUND POWERED
//
// Compute the climber and ground tether force
// I'm not sure why the tether near the ground is so fat, but lets
// assume 27 MY loading on the tether, with an old vehicle mass of
// 20 tonnes * 1 gee force or approximately 200,000 N
// 
// The new vehicle mass is 16 tonnes, so we have more starting force to
// begin with.  4000 kg times 9.75 m/s is 39 kN;  we will cheat and
// assume 44 kN, plus an additional 16 kN as "buffer" so the tether
// does not go slack as it vibrates.  As the vehicle weight drops with
// altitude, we will increase the vibration and tether loading.

   double  rhog  = CRGND*DENS          ; // kg/m tether density at ground
   double  zg    = sound*rhog          ; // kg/s tether impedance at ground
                                         // typical 378.6 kg/s
                                         
   double  gee0  = (MU/(RE*RE)-om2*RE)*KM ; // m/s2 gravity at ground
   double  fvg   = MV*gee0             ;  // N vehicle loading
   double  fmax  = F0+fvg              ;  // N total tether loading

// The maximum power we can transmit from the ground is:
   
   double pg = 0.5 * fmax * fmax / zg    ; // typ 

   printf( "# %12.3e  gee0  \n", gee0   ); // m/s2 gravity
   printf( "# %12.3e  rhog  \n", rhog   ); // kg/m tether density
   printf( "# %12.3e  fvg   \n", fvg    ); // N    vehicle force
   printf( "# %12.3e  fmax  \n", fmax   ); // N    total force
   printf( "# %12.3e  zg    \n", zg     ); // kg/s impedance

// As the climber ascends, it puts less gravitational weight on the tether
// which means that more force is available for acoustic power.  The 
// available power is limited by (1) pg , the acoustic force that can be
// sent up the tether from the ground or (2) the maximum taper-scaled
// force available at climber altitude.
//
// At high altitude, with a climber velocity limit and not much power
// extracted, the power force extracted will be small, and we need not
// reflect back as much force from the wheel above to prevent reflections
// and standing waves in the tether below.   We will compute the power
// extracted, the wave power transmitted to upper climbers, and compute
// the force at each end of the acoustic rectifier to make that happen.

   double  alt                         ; // km altitude
   double  r                           ; // km radius
   double  er                          ; // MJ energy
   double  ar                          ; // km/s²  acceleration
   double  ed                          ; // MJ energy from geosync
   double  tg                          ; // taper from ground
   double  sg                          ; // tether mass to gravity ratio
   double  gg                          ; // m/s²  gravity

   double  dmass                       ; // tonne delta mass  
   double  istep                       ; // km integration index

   double  speed                       ; // m/s acoustic climb rate
   double  power                       ; // m/s3  climb rate scaling
   
   double  pmax                        ; // W energy available from tether force
   double  fv                          ; // N vehicle gravitational force
   double  fp                          ; // N peak tether force left over
   double  z                           ; // kg/s acoustic impedance at altitude
   double  pxtra                       ; // W extra power

   double  hscale = ISTEP / HOUR       ; // km/ks time scaled to step and speed

   double  hours                       ; // hours

//------------------------------------------------------------------------
// altitude loop
//
   double  mdens = CRGND * DENS        ; // kg/m  tonne/km at ground
   double  mstep = ISTEP * mdens       ; // mt / ISTEP at ground
   double  xmass = 0.0                 ; // tonne running mass
   double  fteth = 0.0                 ; // N running force

   for( alt = ALT0 ; alt <= ALT1 ; alt += ISTEP ) {

      r          = alt + RE            ; // km radius
      gg         = KM*(MU/(r*r)-om2*r) ; // m/s²  acceleration down
      er         = 0.5*om2*r*r + MU/r  ; // MJ energy gravitational
      ed         = ee-er               ; // MJ energy from ground
      tg         = exp( ed / YURI )    ; // taper relative to ground
      dmass      = mstep * tg          ; // TONNE additional mass to whole tether
      xmass     += dmass               ; // tether mass below
      fteth     += KM * dmass * gg     ; // N additional force to whole tether

      // compute available power from below

      fv         = gg * MV             ; // N gravitational force on vehicle

      fp         = fmax - fv           ; // N vibration power
      z          = zg * tg             ; // kg/s impedance
      pmax       = 0.5 * fp * fp / z   ; // W power available for climb
      power      = pmax                ; // W power (if not speed limited)

      speed      = power/fv            ; // m/s, if gg < 0 this gets big

      if( speed > SPMAX ) {
         speed   = SPMAX               ;
         power   = speed*fv            ; // could be negative above GEO
      }

      pxtra      = pmax - power        ; // W extra power for climbers above
      hours     += hscale / speed      ; // hours remaining to GEO


      if( fabs(remainder( alt, ASTEP)) < 0.5*ISTEP ) {
         printf( "%10.0f%9.5f%9.5f%9.5f%9.3f%9.3f",
                     alt,  tg,  gg, power/1e6, speed, hours );

// ---- test -------------------------
/*
         printf( "\n" );
         printf( "# %12.3e  dmass ", dmass );
         printf( "# %12.3e  xmass ", xmass );
         printf( "# %12.3e  z     ", z     );
         printf( "# %12.3e  fteth ", fteth );
         printf( "# %12.3e  fmax  ", fmax  );
         printf( "\n" );
         printf( "# %12.3e  fp    ", fp    );
         printf( "# %12.3e  fv    ", fv    );
         printf( "# %12.3e  pmax  ", pmax  );
         printf( "# %12.3e  pxtra ", pxtra );
         printf( "\n" );
*/
// ---- test -------------------------
//
         printf( "%9.3f%9.3f", pxtra/1e6, gg/tg );
         printf( "%9.3f%9.3f%9.3f%9.3f%9.3f\n",
                  hours-RT, hours-2*RT, hours-3*RT, hours-4*RT, hours-6*RT );
      }
   }

   printf("# altitude  taper    grav     power     speed   time \n");
   printf("# km                 m/s2     MW        m/s     hours\n");

   printf("#\n# CNT properties ------\n");

   double strGPa  = YURI * DENS / 1000.0 ;  // GPa

   printf("# strength=%8.4f GPa    density=%8.2f kg/m^3  yuri=%8.4f MYuri\n",
                      strGPa,              DENS,             YURI        );

   // scale to actual cross section at GEO

   double  crgeo  = CRGEO*1E6          ; // (mm)²  cross section
   double  crre   = tge*crgeo          ; // (mm)²  cross section

   double  strgeo = strGPa*crgeo       ; // kN GPa*(mm)²  strength
   double  strapx = tga*strgeo         ; // kN GPa*(mm)²  strength
   double  strre  = tge*strgeo         ; // kN GPa*(mm)²  strength
  
   double  grre   = KM*ae              ; // m/s²  down, positive
   double  grapx  = -KM*aa             ; // m/s²  up,   negative

   double  mre    = strre  / grre      ; // metric tonnes
   double  mteth  = xmass - mapx       ; // metric tonnes

   printf("#\n#                 position:   APEX         GEO          RE\n");
   printf("# Acceleration   m/s²   %10.6f              %10.6f\n",
                                   grapx,          grre       );
   printf("# cross section (mm)²   %10.6f  %10.6f  %10.6f\n",
                               1e6*crapx,  crgeo,  crre       ); 
   printf("# energy         MJ/kg  %10.6f  %10.6f  %10.6f\n",
                                   -ega,   0.0,    ege        );
   printf("# taper                 %10.6f  %10.6f  %10.6f\n",
                                   tga,    1.0,    tge        ); 
   printf("# Strength       kN     %10.2f  %10.2f  %10.2f\n",
                                   strapx, strgeo, strre      ); 
   printf("# Supported mass tonnes %10.2f              %10.3f\n",
                                   mapx,           mre        );
   printf("#\n# Tether mass %8.2f tonnes\n", mteth            );
   printf(   "# Apex   mass %8.2f tonnes\n", mapx             );       
   printf(   "# Total  mass %8.2f tonnes\n", xmass            );       

   return 0 ;
}
